using System;
using System.ComponentModel;
using System.Drawing;
using System.Globalization;
using System.IO;
using System.Security;
using System.Text;
using System.Windows.Forms;

namespace Luschny.Utils.CodeCruiser
{
    public class CruiserForm : Form
    {
        private GroupBox CodeGroup;
        private IContainer components;
        private Button ConvertButton;
        private Button Copyright;
        private RadioButton csCode;
        private Button DestButton;
        private GroupBox DestinationGroup;
        private Label DestinationLabel;
        private string destName;
        private string dirName;
        private Label dstHelp;
        private CheckBox EmbededCss;
        private OpenFileDialog fileDialog;
        private string fileName;
        private FolderBrowserDialog folderDialog;
        private RadioButton javaCode;
        private CheckBox LineNumber;
        private CheckBox LinePrinterPaper;
        private RadioButton SaveToPoolDir;
        private RadioButton SaveToSourceDir;
        private RadioButton SelectDirectory;
        private RadioButton SelectDirTree;
        private RadioButton SelectFile;
        private Button SelectSource;
        private GroupBox SourceGroup;
        private Label SourceLabel;
        private Label srcHelp;
        private bool srcTypeIsDir;
        private Label styHelp;
        private GroupBox StyleOptions;
        private ToolTip toolTip;
        private RadioButton vbCode;

        internal CruiserForm()
        {
            this.InitializeComponent();
            this.srcTypeIsDir = true;
            this.toolTip.AutoPopDelay = 0x1f40;
            this.toolTip.SetToolTip(this.vbCode, "Visual Basic");
            this.toolTip.SetToolTip(this.csCode, "C# CSharp");
            //this.toolTip.SetToolTip(this.jsCode, "JavaScript");
            //this.toolTip.SetToolTip(this.asyCode, "Asymptotic");
            //this.toolTip.SetToolTip(this.grCode, "Groovy");
            this.toolTip.SetToolTip(this.javaCode, "Java");
            this.toolTip.SetToolTip(this.srcHelp, Application.srcOptions);
            this.toolTip.SetToolTip(this.dstHelp, Application.dstOptions);
            this.toolTip.SetToolTip(this.styHelp, Application.formOptions);
            base.StartPosition = FormStartPosition.CenterScreen;
            this.folderDialog = new FolderBrowserDialog();
            this.fileDialog = new OpenFileDialog();
            this.fileDialog.DefaultExt = "*.java";
            this.fileDialog.Filter = "Java code (*.java)|*.java|CSharp code (*.cs)|*.cs|VB code (*.vb)|*.vb|JavaScript (*.js)|*.js|Groovy (*.groovy)|*.groovy|Asymptotic code (*.asy)|*.asy|PowerShell (*.ps1)|*.ps1|All files (*.*)|*.*";
        }

        private void ConvertButton_Click(object sender, EventArgs e)
        {
            if (!this.Panic())
            {
                CruiserState krass = new CruiserState
                {
                    sourcePath = this.SourceLabel.Text,
                    destinationPath = this.DestinationLabel.Text
                };
                if (this.SelectDirTree.Checked)
                {
                    krass.src = CruiserState.SRC.tree;
                }
                else if (this.SelectDirectory.Checked)
                {
                    krass.src = CruiserState.SRC.dir;
                }
                else
                {
                    krass.src = CruiserState.SRC.file;
                }
                if (this.SaveToSourceDir.Checked)
                {
                    krass.dst = CruiserState.DST.same;
                }
                else
                {
                    krass.dst = CruiserState.DST.pool;
                }
                if (this.csCode.Checked)
                {
                    krass.cod = CruiserState.COD.cs;
                }
                else if (this.vbCode.Checked)
                {
                    krass.cod = CruiserState.COD.vb;
                }
                //else if (this.jsCode.Checked)
                //{
                //    krass.cod = CruiserState.COD.js;
                //}
                //else if (this.grCode.Checked)
                //{
                //    krass.cod = CruiserState.COD.groovy;
                //}
                //else if (this.asyCode.Checked)
                //{
                //    krass.cod = CruiserState.COD.asy;
                //}
                else
                {
                    krass.cod = CruiserState.COD.java;
                }
                krass.sty = 0;
                if (this.LineNumber.Checked)
                {
                    krass.sty |= CruiserState.STY.line;
                }
                if (this.LinePrinterPaper.Checked)
                {
                    krass.sty |= CruiserState.STY.paper;
                }
                if (this.EmbededCss.Checked)
                {
                    krass.sty |= CruiserState.STY.embed;
                }
                this.ConvertButton.Enabled = false;
                this.SelectSource.Enabled = false;
                this.DestButton.Enabled = false;
                this.Copyright.Enabled = false;
                this.KrassWork(krass);
                this.ConvertButton.Enabled = true;
                this.SelectSource.Enabled = true;
                this.DestButton.Enabled = true;
                this.Copyright.Enabled = true;
            }
        }

        private void Copyright_Click(object sender, EventArgs e)
        {
            MessageBox.Show(Application.CopyrightMsg, "C O P Y R I G H T", MessageBoxButtons.OK, MessageBoxIcon.Asterisk);
        }

        private void DestButton_Click(object sender, EventArgs e)
        {
            if (this.folderDialog.ShowDialog() == DialogResult.OK)
            {
                this.destName = this.folderDialog.SelectedPath;
                this.DestinationLabel.Text = this.destName;
                this.SaveToPoolDir.Checked = true;
                base.Invalidate();
            }
        }

        protected override void Dispose(bool disposing)
        {
            if (disposing && (this.components != null))
            {
                this.components.Dispose();
            }
            base.Dispose(disposing);
        }

        private static bool FileHandler(TextWriter sw, out int count, string src, string dst, string searchPattern, bool dir, bool tree, bool sameDir)
        {
            StringBuilder builder = new StringBuilder(600);
            count = 0;
            try
            {
                if (dir)
                {
                    SearchOption searchOption = tree ? SearchOption.AllDirectories : SearchOption.TopDirectoryOnly;
                    DirectoryInfo info = new DirectoryInfo(src);
                    foreach (FileInfo info2 in info.GetFiles(searchPattern, searchOption))
                    {
                        builder.Append("<file>\n<src>");
                        builder.Append(info2.FullName);
                        builder.Append("</src>\n<dst>");
                        if (sameDir)
                        {
                            builder.Append(info2.FullName);
                        }
                        else
                        {
                            builder.Append(Path.Combine(dst, info2.Name));
                            builder.Append("." + ((int)count).ToString(CultureInfo.InvariantCulture));
                        }
                        builder.Append(".html</dst>\n</file>\n");
                        count++;
                    }
                }
                else
                {
                    string fileName = Path.GetFileName(src);
                    builder.Append("<file>\n<src>");
                    builder.Append(src);
                    builder.Append("</src>\n<dst>");
                    builder.Append(Path.Combine(dst, fileName));
                    builder.Append(".html</dst>\n</file>\n");
                    count = 1;
                }
                builder.Append("</CodeCruiser>\n");
                sw.Write(builder.ToString());
            }
            catch (SecurityException exception)
            {
                MessageBox.Show(exception.Message, "Please exclude the offendig path from the source path!", MessageBoxButtons.OK, MessageBoxIcon.Hand);
                return true;
            }
            catch (UnauthorizedAccessException exception2)
            {
                MessageBox.Show(exception2.Message, "Please exclude the offendig path from the source path!", MessageBoxButtons.OK, MessageBoxIcon.Hand);
                return true;
            }
            catch (IOException exception3)
            {
                MessageBox.Show(exception3.Message, "P A N I C !", MessageBoxButtons.OK, MessageBoxIcon.Hand);
                return true;
            }
            return false;
        }

        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            this.SourceGroup = new System.Windows.Forms.GroupBox();
            this.srcHelp = new System.Windows.Forms.Label();
            this.SelectSource = new System.Windows.Forms.Button();
            this.SelectDirTree = new System.Windows.Forms.RadioButton();
            this.SelectDirectory = new System.Windows.Forms.RadioButton();
            this.SelectFile = new System.Windows.Forms.RadioButton();
            this.SourceLabel = new System.Windows.Forms.Label();
            this.DestinationGroup = new System.Windows.Forms.GroupBox();
            this.dstHelp = new System.Windows.Forms.Label();
            this.SaveToPoolDir = new System.Windows.Forms.RadioButton();
            this.SaveToSourceDir = new System.Windows.Forms.RadioButton();
            this.DestButton = new System.Windows.Forms.Button();
            this.ConvertButton = new System.Windows.Forms.Button();
            this.StyleOptions = new System.Windows.Forms.GroupBox();
            this.styHelp = new System.Windows.Forms.Label();
            this.EmbededCss = new System.Windows.Forms.CheckBox();
            this.LinePrinterPaper = new System.Windows.Forms.CheckBox();
            this.LineNumber = new System.Windows.Forms.CheckBox();
            this.DestinationLabel = new System.Windows.Forms.Label();
            this.CodeGroup = new System.Windows.Forms.GroupBox();
            this.javaCode = new System.Windows.Forms.RadioButton();
            this.csCode = new System.Windows.Forms.RadioButton();
            this.vbCode = new System.Windows.Forms.RadioButton();
            this.Copyright = new System.Windows.Forms.Button();
            this.toolTip = new System.Windows.Forms.ToolTip(this.components);
            this.SourceGroup.SuspendLayout();
            this.DestinationGroup.SuspendLayout();
            this.StyleOptions.SuspendLayout();
            this.CodeGroup.SuspendLayout();
            this.SuspendLayout();
            // 
            // SourceGroup
            // 
            this.SourceGroup.Controls.Add(this.srcHelp);
            this.SourceGroup.Controls.Add(this.SelectSource);
            this.SourceGroup.Controls.Add(this.SelectDirTree);
            this.SourceGroup.Controls.Add(this.SelectDirectory);
            this.SourceGroup.Controls.Add(this.SelectFile);
            this.SourceGroup.Location = new System.Drawing.Point(28, 25);
            this.SourceGroup.Name = "SourceGroup";
            this.SourceGroup.Size = new System.Drawing.Size(188, 148);
            this.SourceGroup.TabIndex = 0;
            this.SourceGroup.TabStop = false;
            this.SourceGroup.Text = "Source";
            // 
            // srcHelp
            // 
            this.srcHelp.AutoSize = true;
            this.srcHelp.Font = new System.Drawing.Font("Microsoft Sans Serif", 10.8F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.srcHelp.ForeColor = System.Drawing.SystemColors.HotTrack;
            this.srcHelp.Location = new System.Drawing.Point(152, -5);
            this.srcHelp.Name = "srcHelp";
            this.srcHelp.Size = new System.Drawing.Size(21, 24);
            this.srcHelp.TabIndex = 5;
            this.srcHelp.Text = "?";
            // 
            // SelectSource
            // 
            this.SelectSource.Location = new System.Drawing.Point(19, 103);
            this.SelectSource.Name = "SelectSource";
            this.SelectSource.Size = new System.Drawing.Size(134, 28);
            this.SelectSource.TabIndex = 3;
            this.SelectSource.Text = "Select Source";
            this.SelectSource.UseVisualStyleBackColor = true;
            this.SelectSource.Click += new System.EventHandler(this.SrcButton_Click);
            // 
            // SelectDirTree
            // 
            this.SelectDirTree.AutoSize = true;
            this.SelectDirTree.Location = new System.Drawing.Point(19, 76);
            this.SelectDirTree.Name = "SelectDirTree";
            this.SelectDirTree.Size = new System.Drawing.Size(102, 21);
            this.SelectDirTree.TabIndex = 2;
            this.SelectDirTree.Text = "Select Tree";
            this.SelectDirTree.UseVisualStyleBackColor = true;
            this.SelectDirTree.CheckedChanged += new System.EventHandler(this.SelectDirTree_CheckedChanged);
            // 
            // SelectDirectory
            // 
            this.SelectDirectory.AutoSize = true;
            this.SelectDirectory.Checked = true;
            this.SelectDirectory.Location = new System.Drawing.Point(19, 49);
            this.SelectDirectory.Name = "SelectDirectory";
            this.SelectDirectory.Size = new System.Drawing.Size(112, 21);
            this.SelectDirectory.TabIndex = 1;
            this.SelectDirectory.TabStop = true;
            this.SelectDirectory.Text = "Select Folder";
            this.SelectDirectory.UseVisualStyleBackColor = true;
            this.SelectDirectory.CheckedChanged += new System.EventHandler(this.SelectDirectory_CheckedChanged);
            // 
            // SelectFile
            // 
            this.SelectFile.AutoSize = true;
            this.SelectFile.Location = new System.Drawing.Point(19, 22);
            this.SelectFile.Name = "SelectFile";
            this.SelectFile.Size = new System.Drawing.Size(94, 21);
            this.SelectFile.TabIndex = 0;
            this.SelectFile.Text = "Select File";
            this.SelectFile.UseVisualStyleBackColor = true;
            this.SelectFile.CheckedChanged += new System.EventHandler(this.SelectFile_CheckedChanged);
            // 
            // SourceLabel
            // 
            this.SourceLabel.AutoSize = true;
            this.SourceLabel.Location = new System.Drawing.Point(213, 191);
            this.SourceLabel.Name = "SourceLabel";
            this.SourceLabel.Size = new System.Drawing.Size(0, 17);
            this.SourceLabel.TabIndex = 4;
            // 
            // DestinationGroup
            // 
            this.DestinationGroup.Controls.Add(this.dstHelp);
            this.DestinationGroup.Controls.Add(this.SaveToPoolDir);
            this.DestinationGroup.Controls.Add(this.SaveToSourceDir);
            this.DestinationGroup.Controls.Add(this.DestButton);
            this.DestinationGroup.Location = new System.Drawing.Point(244, 25);
            this.DestinationGroup.Name = "DestinationGroup";
            this.DestinationGroup.Size = new System.Drawing.Size(188, 148);
            this.DestinationGroup.TabIndex = 1;
            this.DestinationGroup.TabStop = false;
            this.DestinationGroup.Text = "Destination";
            // 
            // dstHelp
            // 
            this.dstHelp.AutoSize = true;
            this.dstHelp.Font = new System.Drawing.Font("Microsoft Sans Serif", 10.8F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.dstHelp.ForeColor = System.Drawing.SystemColors.HotTrack;
            this.dstHelp.Location = new System.Drawing.Point(152, -5);
            this.dstHelp.Name = "dstHelp";
            this.dstHelp.Size = new System.Drawing.Size(21, 24);
            this.dstHelp.TabIndex = 4;
            this.dstHelp.Text = "?";
            // 
            // SaveToPoolDir
            // 
            this.SaveToPoolDir.AutoSize = true;
            this.SaveToPoolDir.Location = new System.Drawing.Point(19, 67);
            this.SaveToPoolDir.Name = "SaveToPoolDir";
            this.SaveToPoolDir.Size = new System.Drawing.Size(150, 21);
            this.SaveToPoolDir.TabIndex = 3;
            this.SaveToPoolDir.Text = "Save to DestFolder";
            this.SaveToPoolDir.UseVisualStyleBackColor = true;
            this.SaveToPoolDir.CheckedChanged += new System.EventHandler(this.SaveToPoolDir_CheckedChanged);
            // 
            // SaveToSourceDir
            // 
            this.SaveToSourceDir.AutoSize = true;
            this.SaveToSourceDir.Checked = true;
            this.SaveToSourceDir.Location = new System.Drawing.Point(19, 31);
            this.SaveToSourceDir.Name = "SaveToSourceDir";
            this.SaveToSourceDir.Size = new System.Drawing.Size(166, 21);
            this.SaveToSourceDir.TabIndex = 2;
            this.SaveToSourceDir.TabStop = true;
            this.SaveToSourceDir.Text = "Save to SourceFolder";
            this.SaveToSourceDir.UseVisualStyleBackColor = true;
            this.SaveToSourceDir.CheckedChanged += new System.EventHandler(this.SaveToSourceDir_CheckedChanged);
            // 
            // DestButton
            // 
            this.DestButton.Location = new System.Drawing.Point(19, 103);
            this.DestButton.Name = "DestButton";
            this.DestButton.Size = new System.Drawing.Size(134, 28);
            this.DestButton.TabIndex = 1;
            this.DestButton.Text = "Select Destination";
            this.DestButton.UseVisualStyleBackColor = true;
            this.DestButton.Click += new System.EventHandler(this.DestButton_Click);
            // 
            // ConvertButton
            // 
            this.ConvertButton.Location = new System.Drawing.Point(41, 191);
            this.ConvertButton.Name = "ConvertButton";
            this.ConvertButton.Size = new System.Drawing.Size(140, 45);
            this.ConvertButton.TabIndex = 2;
            this.ConvertButton.Text = "Convert";
            this.ConvertButton.UseVisualStyleBackColor = true;
            this.ConvertButton.Click += new System.EventHandler(this.ConvertButton_Click);
            // 
            // StyleOptions
            // 
            this.StyleOptions.Controls.Add(this.styHelp);
            this.StyleOptions.Controls.Add(this.EmbededCss);
            this.StyleOptions.Controls.Add(this.LinePrinterPaper);
            this.StyleOptions.Controls.Add(this.LineNumber);
            this.StyleOptions.Location = new System.Drawing.Point(457, 25);
            this.StyleOptions.Name = "StyleOptions";
            this.StyleOptions.Size = new System.Drawing.Size(188, 148);
            this.StyleOptions.TabIndex = 5;
            this.StyleOptions.TabStop = false;
            this.StyleOptions.Text = "Style";
            // 
            // styHelp
            // 
            this.styHelp.AutoSize = true;
            this.styHelp.Font = new System.Drawing.Font("Microsoft Sans Serif", 10.8F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.styHelp.ForeColor = System.Drawing.SystemColors.HotTrack;
            this.styHelp.Location = new System.Drawing.Point(152, -5);
            this.styHelp.Name = "styHelp";
            this.styHelp.Size = new System.Drawing.Size(21, 24);
            this.styHelp.TabIndex = 3;
            this.styHelp.Text = "?";
            // 
            // EmbededCss
            // 
            this.EmbededCss.AutoSize = true;
            this.EmbededCss.Location = new System.Drawing.Point(24, 103);
            this.EmbededCss.Name = "EmbededCss";
            this.EmbededCss.Size = new System.Drawing.Size(158, 21);
            this.EmbededCss.TabIndex = 2;
            this.EmbededCss.Text = "EmbededStyleSheet";
            this.EmbededCss.UseVisualStyleBackColor = true;
            // 
            // LinePrinterPaper
            // 
            this.LinePrinterPaper.AutoSize = true;
            this.LinePrinterPaper.Checked = true;
            this.LinePrinterPaper.CheckState = System.Windows.Forms.CheckState.Checked;
            this.LinePrinterPaper.Location = new System.Drawing.Point(24, 67);
            this.LinePrinterPaper.Name = "LinePrinterPaper";
            this.LinePrinterPaper.Size = new System.Drawing.Size(137, 21);
            this.LinePrinterPaper.TabIndex = 1;
            this.LinePrinterPaper.Text = "LinePrinterPaper";
            this.LinePrinterPaper.UseVisualStyleBackColor = true;
            // 
            // LineNumber
            // 
            this.LineNumber.AutoSize = true;
            this.LineNumber.Checked = true;
            this.LineNumber.CheckState = System.Windows.Forms.CheckState.Checked;
            this.LineNumber.Location = new System.Drawing.Point(24, 31);
            this.LineNumber.Name = "LineNumber";
            this.LineNumber.Size = new System.Drawing.Size(107, 21);
            this.LineNumber.TabIndex = 0;
            this.LineNumber.Text = "LineNumber";
            this.LineNumber.UseVisualStyleBackColor = true;
            // 
            // DestinationLabel
            // 
            this.DestinationLabel.AutoSize = true;
            this.DestinationLabel.Location = new System.Drawing.Point(213, 219);
            this.DestinationLabel.Name = "DestinationLabel";
            this.DestinationLabel.Size = new System.Drawing.Size(0, 17);
            this.DestinationLabel.TabIndex = 6;
            // 
            // CodeGroup
            // 
            this.CodeGroup.Controls.Add(this.javaCode);
            this.CodeGroup.Controls.Add(this.csCode);
            this.CodeGroup.Controls.Add(this.vbCode);
            this.CodeGroup.Location = new System.Drawing.Point(668, 25);
            this.CodeGroup.Name = "CodeGroup";
            this.CodeGroup.Size = new System.Drawing.Size(106, 148);
            this.CodeGroup.TabIndex = 7;
            this.CodeGroup.TabStop = false;
            this.CodeGroup.Text = "Code";
            // 
            // javaCode
            // 
            this.javaCode.AutoSize = true;
            this.javaCode.Location = new System.Drawing.Point(17, 102);
            this.javaCode.Name = "javaCode";
            this.javaCode.Size = new System.Drawing.Size(63, 21);
            this.javaCode.TabIndex = 2;
            this.javaCode.Text = "JAVA";
            this.javaCode.UseVisualStyleBackColor = true;
            // 
            // csCode
            // 
            this.csCode.AutoSize = true;
            this.csCode.Checked = true;
            this.csCode.Location = new System.Drawing.Point(17, 66);
            this.csCode.Name = "csCode";
            this.csCode.Size = new System.Drawing.Size(47, 21);
            this.csCode.TabIndex = 1;
            this.csCode.TabStop = true;
            this.csCode.Text = "CS";
            this.csCode.UseVisualStyleBackColor = true;
            // 
            // vbCode
            // 
            this.vbCode.AutoSize = true;
            this.vbCode.Location = new System.Drawing.Point(17, 31);
            this.vbCode.Name = "vbCode";
            this.vbCode.Size = new System.Drawing.Size(47, 21);
            this.vbCode.TabIndex = 0;
            this.vbCode.Text = "VB";
            this.vbCode.UseVisualStyleBackColor = true;
            // 
            // Copyright
            // 
            this.Copyright.BackColor = System.Drawing.SystemColors.HotTrack;
            this.Copyright.Location = new System.Drawing.Point(12, 8);
            this.Copyright.Name = "Copyright";
            this.Copyright.Size = new System.Drawing.Size(16, 16);
            this.Copyright.TabIndex = 8;
            this.Copyright.TextAlign = System.Drawing.ContentAlignment.TopCenter;
            this.Copyright.UseVisualStyleBackColor = false;
            this.Copyright.Click += new System.EventHandler(this.Copyright_Click);
            // 
            // CruiserForm
            // 
            this.ClientSize = new System.Drawing.Size(804, 272);
            this.Controls.Add(this.Copyright);
            this.Controls.Add(this.CodeGroup);
            this.Controls.Add(this.DestinationLabel);
            this.Controls.Add(this.StyleOptions);
            this.Controls.Add(this.SourceLabel);
            this.Controls.Add(this.ConvertButton);
            this.Controls.Add(this.DestinationGroup);
            this.Controls.Add(this.SourceGroup);
            this.Name = "CruiserForm";
            this.Text = "CC Code Cruiser";
            this.SourceGroup.ResumeLayout(false);
            this.SourceGroup.PerformLayout();
            this.DestinationGroup.ResumeLayout(false);
            this.DestinationGroup.PerformLayout();
            this.StyleOptions.ResumeLayout(false);
            this.StyleOptions.PerformLayout();
            this.CodeGroup.ResumeLayout(false);
            this.CodeGroup.PerformLayout();
            this.ResumeLayout(false);
            this.PerformLayout();

        }

        private void KrassWork(CruiserState krass)
        {
            string folderPath = Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData);
            string path = folderPath + @"\CoolCodeList.xml";
            string str3 = (krass.dst == CruiserState.DST.same) ? folderPath : krass.destinationPath;
            StringBuilder builder = new StringBuilder(300);
            bool flag = false;
            int count = 0;
            Color backColor = this.ConvertButton.BackColor;
            this.ConvertButton.BackColor = Color.FromArgb(0x99, 0xcc, 0xff);
            this.ConvertButton.Text = string.Format(CultureInfo.InvariantCulture, "Scanning ...", new object[0]);
            this.ConvertButton.Refresh();
            using (StreamWriter writer = new StreamWriter(path))
            {
                builder.Append("<?xml version=\"1.0\" encoding=\"utf-8\" ?>\n");
                builder.Append("<CodeCruiser>\n");
                writer.Write(builder.ToString());
                if (Directory.Exists(krass.sourcePath))
                {
                    flag = FileHandler(writer, out count, krass.sourcePath, krass.destinationPath, "*." + krass.cod.ToString(), true, krass.src == CruiserState.SRC.tree, krass.dst == CruiserState.DST.same);
                }
                else
                {
                    flag = FileHandler(writer, out count, krass.sourcePath, krass.destinationPath, "", false, false, false);
                }
            }
            if (!flag)
            {
                if (count > 0)
                {
                    this.ConvertButton.Text = string.Format(CultureInfo.InvariantCulture, "C o n v e r t i n g \n {0} f i l e (s)", new object[] { count });
                    this.ConvertButton.Refresh();
                    IndexFile index = new IndexFile();
                    try
                    {
                        new XHtmlWriter(krass.cod.ToString(), (krass.sty & CruiserState.STY.paper) != 0, (krass.sty & CruiserState.STY.line) != 0, (krass.sty & CruiserState.STY.embed) != 0).Transform(path, index, new CallBack(this.ConvertButton, count));
                    }
                    catch (Exception exception)
                    {
                        index.Save(str3 + @"\__index.html");
                        MessageBox.Show(exception.Message, "P A N I C !", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                    }
                    if (!flag)
                    {
                        string destDir = str3 + @"\__index.html";
                        string src = index.Save(destDir);
                        destDir = str3 + @"\_index.html";
                        CodeFrame.WriteFrame(destDir, src);
                        if (MessageBox.Show(count + " file(s) processed!\nDo you want to preview the files?\n", "S U C C E S S !", MessageBoxButtons.YesNo, MessageBoxIcon.Asterisk) == DialogResult.Yes)
                        {
                            new BrowserForm(destDir).Show();
                        }
                    }
                }
                else
                {
                    MessageBox.Show("No files of given type found!", "", MessageBoxButtons.OK, MessageBoxIcon.Asterisk);
                }
            }
            this.ConvertButton.BackColor = backColor;
            this.ConvertButton.Text = "Convert";
            this.ConvertButton.Refresh();
        }

        private bool Panic()
        {
            if ((this.SourceLabel.Text == null) || (!File.Exists(this.SourceLabel.Text) && !Directory.Exists(this.SourceLabel.Text)))
            {
                MessageBox.Show("Source file/directory is not valid!", "P A N I C !", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                return true;
            }
            if (File.Exists(this.SourceLabel.Text))
            {
                this.setCodeRadioButton(this.SourceLabel.Text);
            }
            if ((this.DestinationLabel.Text != null) && Directory.Exists(this.DestinationLabel.Text))
            {
                return false;
            }
            MessageBox.Show("Destination directory is not valid!", "P A N I C !", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
            return true;
        }

        private void SaveToPoolDir_CheckedChanged(object sender, EventArgs e)
        {
            RadioButton button = sender as RadioButton;
            if (button.Checked && (this.fileName != null))
            {
                this.DestinationLabel.Text = this.destName;
                this.SaveToPoolDir.Checked = true;
            }
        }

        private void SaveToSourceDir_CheckedChanged(object sender, EventArgs e)
        {
            RadioButton button = sender as RadioButton;
            if (button.Checked)
            {
                if (this.SelectFile.Checked)
                {
                    if (this.fileName != null)
                    {
                        this.dirName = Path.GetDirectoryName(this.fileName);
                        this.destName = this.dirName;
                        this.DestinationLabel.Text = this.dirName;
                    }
                }
                else if (this.dirName != null)
                {
                    this.destName = this.dirName;
                    this.DestinationLabel.Text = this.dirName;
                }
            }
        }

        private void SelectDir(RadioButton b)
        {
            if (b.Checked && !this.srcTypeIsDir)
            {
                if (this.fileName != null)
                {
                    this.dirName = Path.GetDirectoryName(this.fileName);
                    this.SourceLabel.Text = this.dirName;
                }
                else
                {
                    this.SourceLabel.Text = "";
                }
                this.srcTypeIsDir = true;
            }
        }

        private void SelectDirectory_CheckedChanged(object sender, EventArgs e)
        {
            this.SelectDir(sender as RadioButton);
        }

        private void SelectDirTree_CheckedChanged(object sender, EventArgs e)
        {
            this.SelectDir(sender as RadioButton);
        }

        private void SelectFile_CheckedChanged(object sender, EventArgs e)
        {
            RadioButton button = sender as RadioButton;
            if (button.Checked)
            {
                this.srcTypeIsDir = false;
                this.fileName = null;
                this.SourceLabel.Text = "";
            }
        }

        private void SelectFileDialog(object sender, EventArgs e)
        {
            this.setFileDefaultExt();
            if (this.fileDialog.ShowDialog() == DialogResult.OK)
            {
                this.fileName = this.fileDialog.FileName;
            }
        }

        private void SelectFolderDialog(object sender, EventArgs e)
        {
            if (this.folderDialog.ShowDialog() == DialogResult.OK)
            {
                this.dirName = this.folderDialog.SelectedPath;
            }
        }

        private void setCodeRadioButton(string fileName)
        {
            if (fileName != null)
            {
                string str = Path.GetExtension(fileName).TrimStart(new char[] { '.' });
                if (str == CruiserState.COD.java.ToString())
                {
                    this.javaCode.Checked = true;
                }
                else if (str == CruiserState.COD.cs.ToString())
                {
                    this.csCode.Checked = true;
                }
                else if (str == CruiserState.COD.vb.ToString())
                {
                    this.vbCode.Checked = true;
                }
                //else if (str == CruiserState.COD.js.ToString())
                //{
                //    this.jsCode.Checked = true;
                //}
                //else if (str == CruiserState.COD.asy.ToString())
                //{
                //    this.asyCode.Checked = true;
                //}
                //else if (str == CruiserState.COD.groovy.ToString())
                //{
                //    this.grCode.Checked = true;
                //}
            }
        }

        private void setFileDefaultExt()
        {
            if (this.javaCode.Checked)
            {
                this.fileDialog.DefaultExt = "*.java";
            }
            else if (this.csCode.Checked)
            {
                this.fileDialog.DefaultExt = "*.cs";
            }
            else if (this.vbCode.Checked)
            {
                this.fileDialog.DefaultExt = "*.vb";
            }
            //else if (this.jsCode.Checked)
            //{
            //    this.fileDialog.DefaultExt = "*.js";
            //}
            //else if (this.asyCode.Checked)
            //{
            //    this.fileDialog.DefaultExt = "*.asy";
            //}
            //else if (this.grCode.Checked)
            //{
            //    this.fileDialog.DefaultExt = "*.groovy";
            //}
        }

        private void SrcButton_Click(object sender, EventArgs e)
        {
            if (this.SelectFile.Checked)
            {
                this.SelectFileDialog(this, e);
                if (this.SaveToSourceDir.Checked)
                {
                    this.dirName = Path.GetDirectoryName(this.fileName);
                    this.DestinationLabel.Text = this.dirName;
                }
                this.SourceLabel.Text = this.fileName;
                this.setCodeRadioButton(this.fileName);
            }
            else
            {
                this.SelectFolderDialog(sender, e);
                this.fileName = this.dirName;
                this.SourceLabel.Text = this.fileName;
                if (this.SaveToSourceDir.Checked)
                {
                    this.destName = this.dirName;
                    this.DestinationLabel.Text = this.dirName;
                }
            }
        }

        public class CallBack
        {
            private Button button;
            private int count;

            public CallBack(Button button, int count)
            {
                this.button = button;
                this.count = count;
            }

            public void Call()
            {
                this.count--;
                if ((this.count % 7) == 0)
                {
                    this.button.Text = string.Format(CultureInfo.InvariantCulture, "C o n v e r t i n g \n {0} f i l e (s)", new object[] { this.count });
                    this.button.Refresh();
                }
            }
        }
    }
} 
