package de.luschny.struct.genbintree;

/**
 * "Friend" Klasse Binaerer Knoten
 * 
 * Speichert den "Wert" des Knoten
 * und enthaelt die "Zeiger" auf den rechten und linken Teilbaum
 * 
 * @author Peter Luschny
 * @version 1.0, 2001-01-04
 */

class GenericBinaryNode<A>
{
    A value;                      // Wert des Knoten
    GenericBinaryNode<A> left;    // linke  Teilbaumwurzel
    GenericBinaryNode<A> right;   // rechte Teilbaumwurzel

    /**
     * Ein Knoten hat stets einen Wert
     * aber anfaenglich keine Kinder
     * 
     * @param Wert des Knoten
     */

    GenericBinaryNode(A value)
    {
        this.value = value;
    }
}

/**
 * Ein binaerer Baum ist eine Kette von Knoten,
 * wobei jeder Knoten einen Wert (ein Datenelement) enthaelt
 * und Verweise auf die am Knoten haengenden Unterbaeume,
 * von denen ein Knoten maximal zwei haben kann.
 * Betrachtet man die Beziehungen lokal, spricht man auch vom
 * Elternknoten und dem linken bzw. rechten Kindknoten.
 * Jeder Knoten im Baum hat einen eindeutigen Elternknoten
 * - mit einer Ausnahme: die Wurzel des Baums hat keinen Elternknoten.
 * 
 * @author Peter Luschny
 * @version 1.0, 2000-12-10
 */

public class GenericBinaryTree<A>
{
    // BaumWurzel
    private GenericBinaryNode<A> root;

    /**
     * Konstruiert einen (leeren) Binrbaum.
     * 
     */

    public GenericBinaryTree() {}

    /**
     * Liefert die Wurzel des Baums.
     * 
     * @return Baumwurzel
     */

    public Node getRoot()
    {
        Node wurzel = new Node();
        wurzel.node = root;

        return wurzel;
    }

    /**
     * Liefert den linken Kindknoten.
     * 
     * @param node betrachteter Elternknoten
     * 
     * @return Wurzel des linken Teilbaums
     */

    public Node getLeftChild(Node parent)
    {
        Node child = new Node();
        if (parent.isValid())
        {
            child.node = parent.node.left;
        }

        return child;
    }

    /**
     * Liefert den rechten Kindknoten.
     * 
     * @param parent betrachteter Elternknoten
     * 
     * @return Wurzel des rechten Teilbaums
     */

    public Node getRightChild(Node parent)
    {
        Node child = new Node();
        if (parent.isValid())
        {
            child.node = parent.node.right;
        }

        return child;
    }

    /**
     * Setzt den Wert der Wurzel des Baums.
     * 
     * @param value Wert
     */

    public void setRoot(A value)
    {
        if (root == null)
        {
            root = new GenericBinaryNode<A>(value);
        }
    }

    /**
     * Setzt den Wert des linken Kindknotens.
     * 
     * @param value Wert
     * @param parent Elternknoten
     */

    public void setLeftChild(A value, Node parent)
    {
        if (root == null)
        {
            return;
        }

        GenericBinaryNode<A> child = new GenericBinaryNode<A>(value);
        child.right = parent.node.left;
        parent.node.left = child;
    }

    /**
     * Setzt den Wert des rechten Kindknotens.
     * 
     * @param value Wert
     * @param node Elternknoten
     */

    public void setRightChild(A value, Node parent)
    {
        if (root == null)
        {
            return;
        }

        GenericBinaryNode<A> child = new GenericBinaryNode<A>(value);
        child.right = parent.node.right;
        parent.node.right = child;
    }

    /**
     * Loescht alle Knoten im Baum.
     */

    public void clear()
    {
        remove(getRoot());
        root = null;
    }

    /**
     * Loescht den Knoten und alle seine Kinderknoten.
     * 
     * @param parent Wurzel des zu loeschenden Teilbaums
     */

    private void remove(Node parent)
    {
        if (parent.isValid())
        {
            remove(getLeftChild(parent));
            remove(getRightChild(parent));
            parent.delete();
        }
    }

    /**
     * Liefert den Wert eines Knoten.
     * 
     * @param pos betrachteter Knoten
     * 
     * @return Wert des Knoten
     */

    public A get(Node pos)
    {
        if (pos.isValid())
        {
            return pos.node.value;
        }
        else
        {
            throw new NullPointerException();
        }
    }

    /**
     * Ersetzt den Wert des Knoten durch den angereichten Wert.
     * 
     * @param value neuer Wert
     * @param pos zu veraendernder Knoten
     */

    public void replace(A value, Node pos)
    {
        if (pos.isValid())
        {
            pos.node.value = value;
        }
    }

    /**
     * Liefert die Anzahl der Knoten im Teilbaum.
     * 
     * @param node Wurzel des betrachteten Teilbaums
     * 
     * @return Anzahl der Knoten im Teilbaum
     */

    public int size(Node parent)
    {
        if (parent.isValid())
        {
            return size(getLeftChild(parent))
                    + size(getRightChild(parent)) + 1;
        }
        else
        {
            return 0;
        }
    }

    /**
     * Ein Node ist (ein Zeiger auf) ein Grundelement eines Binaeren Baums.
     * Ein Node selbst hat keine oeffentlichen Methoden.
     * Mit ihm kann man also einen Baum nicht veraendern, dazu dienen
     * ausschliesslich die oeffentlichen Methoden des BinaryTree.
     * 
     * Waehrend Microsoft Sprachkonstrukte wie "unsafe pointer" einfuehrt,
     * bedienen wir uns hier also eines "safe pointers".
     * 
     * @author Peter Luschny
     * @version 1.0, 2000-12-10
     */

    public class Node
    {
        // worauf ein Node zeigt
        GenericBinaryNode<A> node;

        /**
         * Erzeugt Zeiger auf einen Knoten eines BinaryTrees.
         */

        Node() {}

        /**
         * Ein Node ist gueltig, wenn er auf ein Baumelement zeigt.
         * 
         * @return das Resultat dieser Pruefung
         */

        final boolean isValid()
        {
            return node != null;
        }

        /**
         * Loescht den Node, macht ihn ungueltig.
         */

        final void delete()
        {
            node = null;
        }
    }
}

